import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Calendar, DollarSign, Bike, MessageSquare } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyRentals.css'

const MyRentals = () => {
  const [rentals, setRentals] = useState({ active: [], past: [] })
  const [activeTab, setActiveTab] = useState('active')

  useEffect(() => {
    loadRentals()
    const interval = setInterval(loadRentals, 500)
    return () => clearInterval(interval)
  }, [])

  const loadRentals = () => {
    const saved = localStorage.getItem('bikeRentals')
    if (saved) {
      try {
        setRentals(JSON.parse(saved))
      } catch (e) {
        setRentals({ active: [], past: [] })
      }
    } else {
      const defaultRentals = {
        active: [
          {
            id: 1,
            model: 'Trek Mountain Pro',
            type: 'Горный',
            date: '2025-03-28',
            returnDate: '2025-04-05',
            price: 1500,
            status: 'active'
          },
          {
            id: 2,
            model: 'City Cruiser 2024',
            type: 'Городской',
            date: '2025-03-25',
            returnDate: '2025-04-02',
            price: 1200,
            status: 'active'
          }
        ],
        past: [
          {
            id: 3,
            model: 'Electric Speed X1',
            type: 'Электровелосипед',
            date: '2025-03-15',
            returnDate: '2025-03-20',
            price: 2000,
            review: 'Отличный велосипед, очень удобный!'
          },
          {
            id: 4,
            model: 'Kids Bike Junior',
            type: 'Детский',
            date: '2025-03-10',
            returnDate: '2025-03-12',
            price: 800,
            review: 'Ребенок в восторге!'
          }
        ]
      }
      setRentals(defaultRentals)
      localStorage.setItem('bikeRentals', JSON.stringify(defaultRentals))
    }
  }

  const stats = [
    { 
      icon: Bike, 
      value: rentals.past.length, 
      label: 'Завершено',
      color: '#8B5CF6'
    },
    { 
      icon: DollarSign, 
      value: rentals.past.reduce((sum, r) => sum + (r.price || 0), 0), 
      label: 'Потрачено',
      color: '#7C3AED'
    },
    { 
      icon: MessageSquare, 
      value: rentals.past.filter(r => r.review).length, 
      label: 'Отзывов',
      color: '#A78BFA'
    }
  ]

  return (
    <div className="my-rentals">
      <div className="container">
        <div className="header-section">
          <h1 className="main-title">Мои аренды</h1>
          <p className="main-subtitle">Управляйте своими арендами велосипедов</p>
        </div>

        <div className="stats-circles">
          {stats.map((stat, index) => (
            <motion.div
              key={stat.label}
              className="stat-circle"
              initial={{ opacity: 0, scale: 0.8 }}
              animate={{ opacity: 1, scale: 1 }}
              transition={{ delay: index * 0.1, duration: 0.5 }}
              whileHover={{ scale: 1.1 }}
            >
              <div className="circle-icon" style={{ borderColor: `${stat.color}40` }}>
                <stat.icon size={24} color={stat.color} />
              </div>
              <div className="circle-content">
                <div className="circle-value">
                  {stat.label === 'Потрачено' ? `${stat.value} ₽` : stat.value}
                </div>
                <div className="circle-label">{stat.label}</div>
              </div>
            </motion.div>
          ))}
        </div>

        <div className="tabs-inline">
          <button
            className={`tab-inline ${activeTab === 'active' ? 'active' : ''}`}
            onClick={() => setActiveTab('active')}
          >
            Активные ({rentals.active.length})
          </button>
          <button
            className={`tab-inline ${activeTab === 'past' ? 'active' : ''}`}
            onClick={() => setActiveTab('past')}
          >
            История ({rentals.past.length})
          </button>
        </div>

        <div className="timeline-container">
          <div className="timeline">
            {activeTab === 'active' ? (
              rentals.active.length > 0 ? (
                rentals.active.map((rental, index) => (
                  <motion.div
                    key={rental.id}
                    className="timeline-item"
                    initial={{ opacity: 0, x: -30 }}
                    animate={{ opacity: 1, x: 0 }}
                    transition={{ delay: index * 0.1 }}
                    whileHover={{ x: 5 }}
                  >
                    <div className="timeline-marker"></div>
                    <div className="timeline-content">
                      <div className="timeline-header">
                        <div className="timeline-bike">
                          <Bike size={20} color="#8B5CF6" />
                        </div>
                        <div className="timeline-info">
                          <h3 className="timeline-model">{rental.model}</h3>
                          <p className="timeline-type">{rental.type}</p>
                        </div>
                        <div className="timeline-price">{rental.price} ₽</div>
                      </div>
                      <div className="timeline-dates">
                        <span>{new Date(rental.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })} - {new Date(rental.returnDate).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })}</span>
                        <span className="timeline-status">Активна</span>
                      </div>
                    </div>
                  </motion.div>
                ))
              ) : (
                <div className="empty-state">
                  <Bike size={56} color="#475569" />
                  <p>У вас нет активных аренд</p>
                </div>
              )
            ) : (
              rentals.past.length > 0 ? (
                rentals.past.map((rental, index) => (
                  <motion.div
                    key={rental.id}
                    className="timeline-item"
                    initial={{ opacity: 0, x: -30 }}
                    animate={{ opacity: 1, x: 0 }}
                    transition={{ delay: index * 0.1 }}
                    whileHover={{ x: 5 }}
                  >
                    <div className="timeline-marker past"></div>
                    <div className="timeline-content">
                      <div className="timeline-header">
                        <div className="timeline-bike">
                          <Bike size={20} color="#8B5CF6" />
                        </div>
                        <div className="timeline-info">
                          <h3 className="timeline-model">{rental.model}</h3>
                          <p className="timeline-type">{rental.type}</p>
                          {rental.review && (
                            <div className="timeline-review">
                              <MessageSquare size={14} color="#8B5CF6" />
                              <span>{rental.review}</span>
                            </div>
                          )}
                        </div>
                        <div className="timeline-price">{rental.price} ₽</div>
                      </div>
                      <div className="timeline-dates">
                        <span>{new Date(rental.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })} - {new Date(rental.returnDate).toLocaleDateString('ru-RU', { day: 'numeric', month: 'short' })}</span>
                      </div>
                    </div>
                  </motion.div>
                ))
              ) : (
                <div className="empty-state">
                  <Bike size={56} color="#475569" />
                  <p>История аренд пуста</p>
                </div>
              )
            )}
          </div>
        </div>

        <div className="slider-wrapper">
          <ImageSlider />
        </div>
      </div>
    </div>
  )
}

export default MyRentals

